<?php

namespace Rawbinn\Larapress\Services;

use Illuminate\Support\Arr;
use Illuminate\Support\Facades\URL;
use Collective\Html\HtmlFacade as HTML;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Request;

/**
 * This class is for admin sidebar menu
 * Class LarapressMenu
 * @package Rawbinn\Larapress\Services
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class LarapressMenu {

    /**     *
     * @var array
     */
    protected $items;

    /**
     * @var string
     */
    protected $current;

    /**
     * @var string
     */
    protected $currentKey;

    /**
     * LarapressMenu constructor
     */
    public function __construct() {
        $this->current = Request::url();
    }

    /*
     * Shortcut method for create a menu with a callback.
     * This will allow you to do things like fire an event on creation.
     * 
     * @param callable $callback Callback to use after the menu creation
     * @return object
     */
    public static function create($callback) {
        $menu = new LarapressMenu();
        $callback($menu);
        $menu->sortItems();
        return $menu;
    }

    /*
     * Add a menu item to the item stack
     * 
     * @param string $key Dot separated hierarchy
     * @param string $name Text for the anchor
     * @param string $url URL for the anchor
     * @param integer $sort Sorting index for the items
     * @param string $icon URL to use for the icon
    */
    public function add($key, $name, $url, $sort = 0, $icon = null, $pattern = null)
    {
        $item = array(
            'key'       => $key,
            'name'      => $name,
            'url'       => $url,
            'sort'      => $sort,
            'icon'      => $icon,
            'pattern'   => $pattern,
            'children'  => array()
        );

        $children = str_replace('.', '.children.', $key);

        Arr::set($this->items, $children, $item);

        if($url == $this->current) {
            $this->currentKey = $key;
        }
    }

    /*
     * Recursive function to loop through items and create a menu
     * 
     * @param array $items List of items that need to be rendered
     * @param boolean $level Which level you are currently rendering
     * @return string
    */
    public function render($items = null, $level = 1)
    {
        $items = $items ?: $this->items;
        $menu = null;
        $firstlevel = ($level === 1) ? 1 : $level;
        if($items!=null){
            foreach($items as $item) {
                if(is_array($item['pattern'])){
                    $uri_pattern = $item['pattern'];
                }else{
                    $uri_pattern = [$item['pattern']];
                }
                if($firstlevel > 1) {
                    $menu .= '<ul class="nav nav-treeview">';
                }
                $classes = [];
                $classes[] = 'nav-item';
                $has_children = sizeof($item['children']);
                if ($has_children) {
                    $classes[] = 'has-treeview';
                }
                if(if_uri_pattern($uri_pattern)) {
                    $classes[] = 'menu-open';
                }
                $menu .= '<li' . HTML::attributes(['class' => implode(' ', $classes)]) . ' >';
                $menu .= $this->createAnchor($item);
                $menu .= ($has_children) ? $this->render($item['children'], ++$level) : '';
                $menu .= '</li>';
                if($firstlevel > 1) {
                    $menu .= '</ul>';
                }
            }

        }
        return $menu;
    }

    /*
     * Method to render an anchor
     * 
     * @param array $item Item that needs to be turned into a link
     * @return string
     */
    private function createAnchor($item)
    {
        $output = '<a class="nav-link '.$this->getActive($item).'" href="' . $item['url'] . '">';
        $output .= $this->createIcon($item);
        $output .= '<p>'.$item['name'];
        $has_children = sizeof($item['children']);
        if ($has_children) {
            $output .= '<i class="fa fa-angle-left right"></i>';
        }
        $output .= '</p>';
            
        $output .= '</a>';

        return $output;
    }

    /*
     * Method to render an icon
     * 
     * @param array $item Item that needs to be turned into a icon
     * @return string
    */
    private function createIcon($item)
    {
        $output = '';

        if($item['icon']) {
            $output .= sprintf(
                '<i class="fas fa-%s"></i> ',
                $item['icon']
            );
        }
        else{
            $output .= sprintf(
                '<i class="far fa-circle"></i> ',
                $item['icon']
            );
        }

        return $output;
    }

    /*
     * Method to sort through the menu items and put them in order
     * 
     * @return void
    */
    private function sortItems() {
        if($this->items!=null){
            usort($this->items, function($a, $b) {
                if($a['sort'] == $b['sort']) {
                    return 0;
                }

                return ($a['sort'] < $b['sort'] ? -1 : 1);
            });
        }
    }

    /*
     * Method to find the active links
     * 
     * @param array $item Item that needs to be checked if active
     * @return string
    */
    private function getActive($item)
    {
        $url = trim($item['url'], '/');

        if ($this->current === $url)
        {
            return 'active';
        }

        if(strpos($this->currentKey, $item['key']) === 0) {
            return 'active';
        }
    }

}