<?php namespace Rawbinn\Larapress\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use TorMorten\Eventy\Facades\Eventy;
use DataTables;
use App\Models\Post\Post;
use Illuminate\Support\Str;
use Rawbinn\Themes\Facades\Theme;
use App\Repositories\Backend\CategoryRepository;

/**
 * Class CustomPostTypeController
 * @package Rawbinn\Larapress\Controllers
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class CustomPostTypeController extends Controller
{
     /**
     * @var CategoryRepository
     */
    protected $category;

    /**
     * BlogController constructor.
     */
    public function __construct(CategoryRepository $category)
    {
        $this->category = $category;
    }

    public function index($type)
    {
        $type = Str::slug(strtolower($type));
        $post_types = get_post_types();
        
        if(!isset($post_types[$type])) {
            abort(404);
        }
        $data = $post_types[$type];
        $data['post_type'] = $type;
        return view('admin.custom_post.index', $data);
    }

    public function create($type)
    {
        $type = Str::slug(strtolower($type));
        $post_types = get_post_types();
        if(!isset($post_types[$type])) {
            abort(404);
        }
        $data = $post_types[$type];
        $data['post_type'] = $type;
        if(array_keys($data['supports']) === range(0, count($data['supports']) - 1)) {
         foreach($data['supports'] as $support) {
             $data['supports'][$support] = ucfirst($support);
             if($support == 'editor') {
                 $data['supports'][$support] = 'Content';
             } elseif($support == 'thumbnail') {
                 $data['supports'][$support] = 'Featured';
             }
         }   
        }
        if( array_key_exists('category', $data['supports'])) {
            $data['categories'] =  $this->category->getTree($type);
        }
        return view('admin.custom_post.create', $data);
    }

    /**
     * @return mixed
     */
    public function store($type, Request $request)
    {
        $type = Str::slug(strtolower($type));
        $post_types = get_post_types();
        if(!isset($post_types[$type])) {
            abort(404);
        }
        $inputs = $request->validate([
            'title' => 'required',
            'subtitle' => 'nullable',
            'content' => 'nullable',
            'meta_title' => 'nullable',
            'meta_description' => 'nullable',
            'featured_image' => 'nullable'
        ]);
        $data = [
            "user_id" => auth()->id(),
            "meta_title" => $inputs['meta_title'] ?? null,
            "meta_description" => $inputs['meta_description'] ?? null,
            "media_id" => $inputs['featured_image'] ?? null,
            "type" => $type
        ];
        $model = new Post();
        foreach(locales() as $locale) {
            foreach ($model->translatedAttributes as $attribute) {
                if (isset($inputs[$attribute])) {
                    $data[$locale][$attribute] = $inputs[$attribute];
                }
            }
        }
        $model = $model->fill($data);
        $model->save();
        $categories = $request->has('category') ? $request->category : 1;
        $model->categories()->attach($categories);
        if ($request->has('saveEdit')) {
            return redirect()->route('admin.custom.post.type.edit', [$type, $model->id])->withFlashSuccess(trans('alerts.backend.general.created', ['attribute' => $type]).' <a href="'.$model->permalink().'" style="color: #fff81a;" target="_blank">View '.ucfirst(strtolower($type)).'</a>');
        }

        return redirect()->route('admin.custom.post.type', $type)->withFlashSuccess(trans('alerts.backend.general.created', ['attribute' => $type]).' <a href="'.$model->permalink().'" style="color: #fff81a;" target="_blank">View '.ucfirst(strtolower($type)).'</a>');
    }

    /**
     * @return View
     */
    public function edit($type, Post $page, Request $request)
    {
        $type = Str::slug(strtolower($type));
        $post_types = get_post_types();
        if(!isset($post_types[$type])) {
            abort(404);
        }
        $data = $post_types[$type];
        $data['lang'] = lang($request->lang);
        $data['post_type'] = $type;
        $data['page'] = $page;

        if(array_keys($data['supports']) === range(0, count($data['supports']) - 1)) {
            foreach($data['supports'] as $support) {
                $data['supports'][$support] = ucfirst($support);
                if($support == 'editor') {
                    $data['supports'][$support] = 'Content';
                } elseif($support == 'thumbnail') {
                    $data['supports'][$support] = 'Featured';
                }
            }   
           }
           
        if(array_key_exists('category', $data['supports'])) {
            $data['categories'] =  $this->category->getTree($type);
        }
        return view('admin.custom_post.edit', $data);
    }

    /**
     * @return mixed
     */
    public function update($type, Post $page, Request $request)
    {
        $type = Str::slug(strtolower($type));
        $post_types = get_post_types();
        if(!isset($post_types[$type])) {
            abort(404);
        }
        $locale = $request->lang ?? app()->getLocale();
        $inputs = $request->validate([
            $locale.'.title' => 'required',
            $locale.'.subtitle' => 'nullable',
            $locale.'.content' => 'nullable',
            'meta_title' => 'nullable',
            'meta_description' => 'nullable',
            'featured_image' => 'nullable',
            'lang' => 'required'
        ]);

        $attributes = [
            "meta_title" => $inputs['meta_title'] ?? null,
            "meta_description" => $inputs['meta_description'] ?? null,
            "media_id" => $inputs['featured_image'] ?? null,
        ];
        $attributes[$inputs['lang']] = $inputs[$inputs['lang']];
        $page->update($attributes);

        $data = $post_types[$type];
        $data['post_type'] = $type;
        
        $categories = $request->has('category') ? $request->category : 1;
        $page->categories()->sync($categories);
        return redirect()->back()->withFlashSuccess(trans('alerts.backend.general.updated', ['attribute' => $type]).' <a href="'.$page->permalink().'" style="color: #fff81a;" target="_blank">View Page</a>');
    }

    /**
     * @param Post page
     *
     * @return mixed
     */
    public function destroy($type, Post $page, Request $request)
    {
        if ($page->delete()) {
            return redirect()->route('admin.custom.post.type', $type)->withFlashSuccess(trans('alerts.backend.general.deleted', ['attribute' => $type]));
        }
    }

    public function destroySelected(Request $request)
    {
        if (!$request->ajax()) {
            exit;
        }
        foreach ($request->id as $id) {
            $page = Post::find($id);
            if ($page != null) {
                $page->delete();
            }
        }

        return response()->json(['status' => true]);
    }

    /**
     * @param null
     *
     * @return view
     */
    public function deleted($type)
    {
        $type = Str::slug(strtolower($type));
        $post_types = get_post_types();
        if(!isset($post_types[$type])) {
            abort(404);
        }
        $data = $post_types[$type];
        $data['post_type'] = $type;
        
        return view('admin.custom_post.deleted', $data);
    }

    /**
     * @return mixed
     */
    public function delete($type, Post $deletedPage, Request $request)
    {
        $deletedPage->deleteTranslations();
        $this->pages->forceDelete($deletedPage);
        return redirect()->route('admin.page.deleted')->withFlashSuccess(trans('alerts.backend.general.deleted_permanently', ['attribute' => $type]));
    }

    public function deleteSelected(Request $request)
    {
        if (!$request->ajax()) {
            exit;
        }
        if (!$request->has('id')) {
            return response()->json(['status' => false]);
        }
        foreach ($request->id as $id) {
            $page = Post::withTrashed()->where('id', $id)->first();
            if ($page != null) {
                $page->deleteTranslations();
                $page->forceDelete();
            }
        }

        return response()->json(['status' => true]);
    }

    /**
     * @return mixed
     */
    public function restore($type, $page_id, Request $request)
    {
        $page = Post::withTrashed()->where('id', $page_id)->firstOrFail();
        $page->restore();

        return redirect()->route('admin.custom.post.type', $type)->withFlashSuccess(trans('alerts.backend.general.restored', ['attribute' => $type]));
    }

    /**
     * @return mixed
     */
    public function getForDatatable(Request $request, $type)
    {
        $posts = Post::where('type', Str::slug(strtolower($type)))->where('status', '1');

        if (!auth()->user()->hasRole(1)) {
            $posts->where('user_id', auth()->id());
        }
        if($request->get('trashed') != 'false'){
           $posts =  $posts->onlyTrashed();
        }
        return DataTables::of($posts)
            ->editColumn('title', function ($page) {
                return '<a href="'.route('post', $page->slug).'" target="_blank">'.$page->title.'</a>';
            })
            ->addColumn('author', function ($page) {
                return $page->user->name;
            })
            ->editColumn('created_at', function ($page) {
                return $page->created_at->diffForHumans();
            })
            ->editColumn('updated_at', function ($page) {
                return $page->updated_at->diffForHumans();
            })
            ->addColumn('actions', function ($page) use($type, $request){
                if($request->get('trashed') != 'false') {
                    return '<a href="'.route('admin.custom.post.type.restore', [$type, $page->id]).'" name="restore_page" class="btn btn-xs btn-info"><i class="fa fa-redo" data-toggle="tooltip" data-placement="top" title="'.trans('buttons.backend.blog.restore_blog').'"></i></a> '.'<a href="'.route('admin.custom.post.type.delete-permanently', [$type, $page->id]).'" name="delete_page_perm" class="btn btn-xs btn-danger"><i class="fa fa-trash" data-toggle="tooltip" data-placement="top" title="'.trans('buttons.backend.general.delete_permanently').'"></i></a> ';
                }
                return '<a href="'.route('post', $page->slug).'" target="_blank" class="btn btn-xs btn-info"><i class="fa fa-eye" data-toggle="tooltip" data-placement="top" title="'.trans('buttons.general.crud.view').'"></i></a> '.'<a href="'.route('admin.custom.post.type.edit', [$type, $page->id]).'" class="btn btn-xs btn-primary"><i class="fa fa-pencil-alt" data-toggle="tooltip" data-placement="top" title="'.trans('buttons.general.crud.edit').'"></i></a>'.' <a href="'.route('admin.custom.post.type.destroy', [$type, $page->id]).'"
                data-method="delete"
                data-trans-button-cancel="'.trans('buttons.general.cancel').'"
                data-trans-button-confirm="'.trans('buttons.general.crud.delete').'"
                data-trans-title="'.trans('strings.backend.general.are_you_sure').'"
                class="btn btn-xs btn-danger"><i class="fa fa-trash" data-toggle="tooltip" data-placement="top" title="Trash"></i></a> ';
            })
            ->addColumn('select', function ($page) {
                return '<input type="checkbox" name="id[]" value="'.$page->id.'">';
            })
            ->escapeColumns([])
            ->make(true);
    }
}