<?php namespace App\Services;

use \ZipArchive;
use App\Exceptions\GeneralException;

/**
 * Class Zipper
 * @package App\Services
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class Zipper
{
    /**
     * zip file name
     *
     * @var string
     */
    private $zip_file = null;

    /**
     * Mask for the extraction folder
     *
     * @var int
     */
    private $mask = 0777;

    /**
     * Array of well known zip status codes
     *
     * @var array
     */
    private static $zip_status_codes = Array(
        ZipArchive::ER_OK           => 'No error',
        ZipArchive::ER_MULTIDISK    => 'Multi-disk zip archives not supported',
        ZipArchive::ER_RENAME       => 'Renaming temporary file failed',
        ZipArchive::ER_CLOSE        => 'Closing zip archive failed',
        ZipArchive::ER_SEEK         => 'Seek error',
        ZipArchive::ER_READ         => 'Read error',
        ZipArchive::ER_WRITE        => 'Write error',
        ZipArchive::ER_CRC          => 'CRC error',
        ZipArchive::ER_ZIPCLOSED    => 'Containing zip archive was closed',
        ZipArchive::ER_NOENT        => 'No such file',
        ZipArchive::ER_EXISTS       => 'File already exists',
        ZipArchive::ER_OPEN         => 'Can\'t open file',
        ZipArchive::ER_TMPOPEN      => 'Failure to create temporary file',
        ZipArchive::ER_ZLIB         => 'Zlib error',
        ZipArchive::ER_MEMORY       => 'Malloc failure',
        ZipArchive::ER_CHANGED      => 'Entry has been changed',
        ZipArchive::ER_COMPNOTSUPP  => 'Compression method not supported',
        ZipArchive::ER_EOF          => 'Premature EOF',
        ZipArchive::ER_INVAL        => 'Invalid argument',
        ZipArchive::ER_NOZIP        => 'Not a zip archive',
        ZipArchive::ER_INTERNAL     => 'Internal error',
        ZipArchive::ER_INCONS       => 'Zip archive inconsistent',
        ZipArchive::ER_REMOVE       => 'Can\'t remove file',
        ZipArchive::ER_DELETED      => 'Entry has been deleted'
    );

    /**
     * Class constructor
     *
     * @param   string  $zip_file   ZIP file name
     *
     */

    public function __construct($zip_file) {

        if ( empty($zip_file) ) throw new GeneralException(self::getStatus(ZipArchive::ER_NOENT));

        $this->zip_file = $zip_file;

    }

    /**
     * Open a zip archive
     *
     * @param string  $zip_file   ZIP file name
     *
     * @return Zip
     */
    public static function open($zip_file) {

        try {

            $zip = new Zipper($zip_file);
            $zip->setArchive(self::openZipFile($zip_file));

        } catch (GeneralException $ze) {

            throw $ze;

        }

        return $zip;

    }

    /**
     * Open a zip file
     *
     * @param   string $zip_file   ZIP status code
     * @param   int    $flags      ZIP status code
     *
     * @return  \ZipArchive
     */
    private static function openZipFile($zip_file, $flags = null) {

        $zip = new ZipArchive();

        $open = $zip->open($zip_file, $flags);
        
        if ( $open !== true ) throw new GeneralException(self::getStatus($open));

        return $zip;

    }

    /**
     * Set the current ZipArchive object
     *
     * @param   \ZipArchive     $zip
     *
     * @return  \ZanySoft\Zip\Zip
     */
    final public function setArchive(ZipArchive $zip) {

        $this->zip_archive = $zip;

        return $this;

    }

    /**
     * Get status from zip status code
     *
     * @param   int $code   ZIP status code
     *
     * @return  string
     */
    private static function getStatus($code) {

        if ( array_key_exists($code, self::$zip_status_codes) ) return self::$zip_status_codes[$code];

        else return sprintf('Unknown status %s', $code);

    }

    /**
     * Extract files from zip archive
     *
     * @param   string  $destination    Destination path
     * @param   mixed   $files          (optional) a filename or an array of filenames
     *
     * @return  bool
     */
    public function extract($destination, $files = null) {

        if ( empty($destination) ) throw new GeneralException('Invalid destination path');

        if ( !file_exists($destination) ) {

            $omask = umask(0);

            $action = mkdir($destination, $this->mask, true);

            umask($omask);

            if ( $action === false ) throw new GeneralException("Error creating folder ".$destination);

        }

        if ( !is_writable($destination) ) throw new GeneralException('Destination path not writable');
        
        if (!is_array($files) && !is_string($files) && !is_null($files)) {
            throw new GeneralException('Invalid extractable files');
        }
        
	    $files = is_string($files) ? [$files]: $files;

        if (@sizeof($files) != 0 ) {
            $file_matrix = $files;
        } else {
            $file_matrix = $this->getArchiveFiles();
        }

        $extract = $this->zip_archive->extractTo($destination, $file_matrix);

        if ( $extract === false ) throw new GeneralException(self::getStatus($this->zip_archive->status));

        return true;

    }

    /**
     * Get a list of files in archive (array)
     *
     * @return  array
     */
    public function listFiles() {

        $list = Array();

        for ( $i = 0; $i < $this->zip_archive->numFiles; $i++ ) {

            $name = $this->zip_archive->getNameIndex($i);

            if ( $name === false ) throw new GeneralException(self::getStatus($this->zip_archive->status));

            array_push($list, $name);

        }

        return $list;

    }

    /**
     * Get a list of file contained in zip archive before extraction
     *
     * @return  array
     */
    private function getArchiveFiles() {

        $list = array();

        for ( $i = 0; $i < $this->zip_archive->numFiles; $i++ ) {

            $file = $this->zip_archive->statIndex($i);

            if ( $file === false ) continue;

            $name = str_replace('\\', '/', $file['name']);

            if ( $name[0] == "." AND in_array($this->skip_mode, array("HIDDEN", "ALL")) ) continue;

            if ( $name[0] == "." AND @$name[1] == "_" AND in_array($this->skip_mode, array("ZANYSOFT", "ALL")) ) continue;

            array_push($list, $name);

        }

        return $list;

    }

    /**
     * Close the zip archive
     *
     * @return  bool
     */
    public function close() {

        if ( $this->zip_archive->close() === false ) throw new GeneralException(self::getStatus($this->zip_archive->status));

        return true;

    }

    public function locateName($file)
    {
        return $this->zip_archive->locateName($file);
    }

    public function getNameIndex($index)
    {
        return $this->zip_archive->getNameIndex($index);
    }
}
