<?php

namespace App\Services;

use Illuminate\Support\Facades\Request;

/**
 * Class Tree
 * @version 1.0.0
 * @author Rawbinn Shrestha <contact@rawbinn.com>
 * @organization RAWBINN.COM.
 */
class Tree {

    /**
     * Contains tree item
     *
     * @var array
     */
	public $items = [];

    /**
     * Contains acl roles
     *
     * @var array
     */
	public $roles = [];

    /**
     * Contains current item route
     *
     * @var string
     */
	public $current;

    /**
     * Contains current item key
     *
     * @var string
     */
	public $currentKey;

    /**
     * Create a new instance.
     *
     * @return void
     * @author Rawbinn Shrestha ( rawbinnn@gmail.com )
     */
	public function __construct()
	{
		$this->current = Request::url();
	}

	/**
	 * Shortcut method for create a Config with a callback.
	 * This will allow you to do things like fire an event on creation.
	 *
	 * @param  callable  $callback Callback to use after the Config creation
	 * @return object
     * @author Rawbinn Shrestha ( rawbinnn@gmail.com )
	 */
	public static function create($callback = null)
	{
		$tree = new Tree();

		if ($callback) {
			$callback($tree);
		}

		return $tree;
	}

    /**
     * Add a Config item to the item stack
     *
     * @param $item
     * @param string $type
     * @author Rawbinn Shrestha ( rawbinnn@gmail.com )
     */
	public function add($item, $type = '')
	{
        $item['children'] = [];

		if ($type == 'menu') {
            if(strpos($item['route'], '/')!==false) {
                $item['url'] = url($item['route']);
            }else {
                $item['url'] = route($item['route'], $item['params'] ?? []);
            }

			if (strpos($this->current, $item['url']) !== false) {
                $this->currentKey = $item['key'];
			}
		} elseif ($type == 'acl') {
			$item['name'] = $item['name'];

			$this->roles[$item['route']] = $item['key'];
		}

		$children = str_replace('.', '.children.', $item['key']);

		$this->array_set($this->items, $children, $item);
	}

    /**
     * Method to find the active links
     *
     * @param $item
     * @return string
     * @author Rawbinn Shrestha ( rawbinnn@gmail.com )
     */
	public function getActive($item)
	{
		$url = trim($item['url'], '/');

		if ((strpos($this->current, $url) !== false) || (strpos($this->currentKey, $item['key']) === 0)) {
			return 'active';
		}
	}

    /**
     * @param $key
     * @param $value
     *
     * @return mixed
     *
     * @author Rawbinn Shrestha ( rawbinnn@gmail.com )
     */
    protected function array_set(&$array, $key, $value)
    {
        if (is_null($key)) {
            return $array = $value;
        }

        $keys = explode('.', $key);
        $count = count($keys);

        while (count($keys) > 1) {
            $key = array_shift($keys);

            if (!isset($array[$key]) || !is_array($array[$key])) {
                $array[$key] = [];
            }

            $array = &$array[$key];
        }

        $finalKey = array_shift($keys);

        if (isset($array[$finalKey])) {
            $array[$finalKey] = $this->arrayMerge($array[$finalKey], $value);
        } else {
            $array[$finalKey] = $value;
        }

        return $array;
    }
    
    /**
     * @return array
     *
     * @author Rawbinn Shrestha ( rawbinnn@gmail.com )
     */
    protected function arrayMerge(array &$array1, array &$array2)
    {
        $merged = $array1;

        foreach ($array2 as $key => &$value) {
            if (is_array($value) && isset($merged[$key]) && is_array($merged[$key])) {
                $merged[$key] = $this->arrayMerge($merged[$key], $value);
            } else {
                $merged[$key] = $value;
            }
        }

        return $merged;
    }

    /**
     * Method to sort through the acl items and put them in order.
     *
     * @param $items
     *
     * @return array
     *
     * @author Rawbinn Shrestha ( rawbinnn@gmail.com )
     */
    public function sortItems($items)
    {
        foreach ($items as &$item) {
            if (count($item['children'])) {
                $item['children'] = $this->sortItems($item['children']);
            }
        }

        usort($items, function ($a, $b) {
            if ($a['sort'] == $b['sort']) {
                return 0;
            }

            return ($a['sort'] < $b['sort']) ? -1 : 1;
        });

        return $this->convertToAssociativeArray($items);
    }

    /**
     * @param $items
     *
     * @return mixed
     *
     * @author Rawbinn Shrestha ( rawbinnn@gmail.com )
     */
    public function convertToAssociativeArray($items)
    {
        foreach ($items as $key1 => $level1) {
            unset($items[$key1]);
            $items[$level1['key']] = $level1;

            if (count($level1['children'])) {
                foreach ($level1['children'] as $key2 => $level2) {
                    $temp2 = explode('.', $level2['key']);
                    $finalKey2 = end($temp2);
                    unset($items[$level1['key']]['children'][$key2]);
                    $items[$level1['key']]['children'][$finalKey2] = $level2;

                    if (count($level2['children'])) {
                        foreach ($level2['children'] as $key3 => $level3) {
                            $temp3 = explode('.', $level3['key']);
                            $finalKey3 = end($temp3);
                            unset($items[$level1['key']]['children'][$finalKey2]['children'][$key3]);
                            $items[$level1['key']]['children'][$finalKey2]['children'][$finalKey3] = $level3;
                        }
                    }
                }
            }
        }

        return $items;
    }
}

