<?php
namespace App\Repositories\Frontend;

use DB;
use Log;
use Dotenv\Dotenv;

/**
 * Class InstallRepository
 * @package App\Repositories\Frontend
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class InstallRepository
{

    public function getRequirements()
    {
        $requirements = [
            'PHP Version (>= 7.1.X)' => version_compare(phpversion(), '7.1', '>='),
            'OpenSSL Extension' => extension_loaded('openssl'),
            'PDO Extension' => extension_loaded('PDO'),
            'PDO MySQL Extension' => extension_loaded('pdo_mysql'),
            'Mbstring Extension' => extension_loaded('mbstring'),
            'Tokenizer Extension' => extension_loaded('tokenizer'),
            'GD Extension' => extension_loaded('gd'),
            'Fileinfo Extension' => extension_loaded('fileinfo'),
            'CURL Extension' => extension_loaded('curl'),
            '.env file' => is_writable(base_path('.env')),
        ];

        if (extension_loaded('xdebug')) {
            $requirements['Xdebug Max Nesting Level (>= 500)'] = (int) ini_get('xdebug.max_nesting_level') >= 500;
        }

        return $requirements;
    }

    public function allRequirementsLoaded()
    {
        $allLoaded = true;

        foreach ($this->getRequirements() as $loaded) {
            if ($loaded == false) {
                $allLoaded = false;
            }
        }

        return $allLoaded;
    }

    public function getPermissions()
    {
        return [
            'storage/app' => is_writable(storage_path('app')),
            'storage/framework/cache' => is_writable(storage_path('framework/cache')),
            'storage/framework/sessions' => is_writable(storage_path('framework/sessions')),
            'storage/framework/views' => is_writable(storage_path('framework/views')),
            'storage/logs' => is_writable(storage_path('logs')),
            'storage' => is_writable(storage_path('')),
            'bootstrap/cache' => is_writable(base_path('bootstrap/cache')),
        ];
    }

    public function allPermissionsGranted()
    {
        $allGranted = true;

        foreach ($this->getPermissions() as $permission => $granted) {
            if ($granted == false) {
                $allGranted = false;
            }
        }

        return $allGranted;
    }

    public function getDisablePermissions()
    {
        return [
            'Base Directory' => !is_writable(base_path('/')),
        ];
    }

    public function allDisablePermissionsGranted()
    {
        $allNotGranted = true;

        foreach ($this->getDisablePermissions() as $permission => $granted) {
            if ($granted == true) {
                $allNotGranted = false;
            }
        }

        return $allNotGranted;
    }

    public function dbCredentialsAreValid($credentials)
    {
        $this->setDatabaseCredentials($credentials);
        $this->reloadEnv();

        try {
            DB::statement("SHOW TABLES");
        } catch (\Exception $e) {
            
            Log::info($e->getMessage());
            return false;
        }

        return true;
    }

    private function reloadEnv()
	{
        $dotenv = Dotenv::createMutable(base_path());
        $dotenv->load();
	}

    public function dbDropSettings()
    {
        try {
            // DB::statement("DROP TABLE `settings`;");
        } catch (\Exception $e) {
            Log::info($e->getMessage());
            return false;
        }

        return true;
    }

    public function setDatabaseCredentials($credentials)
	{
		$default = config('database.default');

		config([
			"database.connections.{$default}.host"     => $credentials['host'],
			"database.connections.{$default}.database" => $credentials['database'],
			"database.connections.{$default}.username" => $credentials['username'],
			"database.connections.{$default}.password" => $credentials['password']
		]);

        $path = base_path('.env');
        $env = file($path);
        $env = str_replace('APP_URL='.env('APP_URL'), 'APP_URL='.url('/'), $env);
        $env = str_replace('DB_HOST='.env('DB_HOST'), 'DB_HOST='.$credentials['host'], $env);
        $env = str_replace('DB_DATABASE='.env('DB_DATABASE'), 'DB_DATABASE='.$credentials['database'], $env);
        $env = str_replace('DB_USERNAME='.env('DB_USERNAME'), 'DB_USERNAME='.$credentials['username'], $env);
        $env = str_replace('DB_PASSWORD='.env('DB_PASSWORD'), 'DB_PASSWORD='.$credentials['password'], $env);

        file_put_contents($path, $env);
    }
}