<?php

namespace App\Repositories\Backend;

use App\Events\Backend\Page\PageDeleted;
use App\Events\Backend\Page\PageUpdated;
use App\Exceptions\GeneralException;
use App\Models\Post\Post;
use App\Repositories\Repository;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use TorMorten\Eventy\Facades\Eventy;

/**
 * Class PageRepository.
 *
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class PageRepository extends Repository
{
    /**
     * Associated Repository Model.
     */
    const MODEL = Post::class;

    /**
     * @param bool $trashed
     *
     * @return mixed
     */
    public function getForDataTable($trashed = false)
    {
        /**
         * Note: You must return deleted_at or the User getActionButtonsAttribute won't
         * be able to differentiate what buttons to show for each row.
         */
        $dataTableQuery = $this->query()
            ->select([
                'id',
                'user_id',
                'slug',
                'type',
                'template',
                'created_at',
                'updated_at',
                'deleted_at',
            ])
            ->where('type', 'page')
            ->orderBY('id', 'desc');

        if (!auth()->user()->hasRole(1)) {
            $dataTableQuery->where('user_id', auth()->id());
        }

        if ($trashed == 'true') {
            return $dataTableQuery->onlyTrashed();
        }

        return $dataTableQuery;
    }

    /**
     * @return mixed
     */
    public function getAll($orderBy = 'ASC')
    {
        return $this->query()->where('type', 'page')->where('status', '1')->orderBy('id', $orderBy)->get();
    }

    /**
     * @param $data
     *
     * @throws GeneralException
     */
    public function create($attributes)
    {
        $data = [
            'type' => 'page',
            'media_id' => ($attributes['featured_image'] != '') ? $attributes['featured_image'] : null,
            'user_id' => access()->id(),
            'template' => $attributes['template'],
            'meta_title' => ($attributes['meta_title'] != '') ? $this->filterString($attributes['meta_title']) : $this->filterString($attributes['title']),
            'meta_description' => ($attributes['meta_description'] != '') ? $this->filterString($attributes['meta_description'], 160) : $this->filterString($attributes['content'] ?? '', 160)
        ];
        $model = self::MODEL;
        $model = new $model();
        try {
            foreach (locales() as $locale) {
                foreach ($model->translatedAttributes as $attribute) {
                    if (isset($attributes[$attribute])) {
                        $data[$locale][$attribute] = $attributes[$attribute];
                    }
                }
            }
            $model->fill($data);
            parent::save($model);
            $sort = 1;
            if (isset($attributes['post_meta']) && is_array($attributes['post_meta'])) {
                foreach ($attributes['post_meta'] as $key => $post) {
                    foreach(locales() as $locale) {
                        $model->meta()->create([
                            'key' => $key,
                            'value' => json_encode($post),
                            'locale' => $locale,
                            'sort' => $sort
                        ]);
                    }
                    $sort = $sort+1;
                }
            }
        } catch (\Exception $e) {
            dd($e->getMessage());
            throw new GeneralException(trans('exceptions.backend.page.create_error'));
        }
        event(new PageUpdated($model, $attributes));
        return $model;
    }

    public function update(Model $page, array $attributes)
    {
        if (!auth()->user()->hasRole(1) && auth()->id() != $page->user_id) {
            throw new GeneralException('You do not have permission to edit this page.');
        }

        
        $data = [
            'slug' => $attributes['slug'],
            'media_id' => ($attributes['media_id'] != '') ? $attributes['media_id'] : null,
            'template' => $attributes['template'],
            'meta_title' => ($attributes['meta_title'] != '') ? $this->filterString($attributes['meta_title']) : $this->filterString($attributes['title']),
            'meta_description' => ($attributes['meta_description'] != '') ? $this->filterString($attributes['meta_description'], 160) : $this->filterString($attributes['content'] ?? '', 160)
        ];
        $data[$attributes['lang']] = $attributes[$attributes['lang']];
       if (parent::update($page, $data)) {
            if (isset($attributes['post_meta']) && is_array($attributes['post_meta'])) {
                $sort = 1;
                // dd($attributes['post_meta']);
                foreach ($attributes['post_meta'] as $key => $cols) {
                    // dd($cols);
                    $meta = $page->metas()->where('locale', $attributes['lang'])->where('key', $key)->first();
                    $page->deleteMetaByKey($key, $attributes['lang']);
                    foreach(locales() as $locale) {
                        if($attributes['lang'] == $locale) {
                            $page->meta()->create([
                                'key' => $key,
                                'value' => json_encode($cols),
                                'sort' => $meta ? $meta->sort : $sort,
                                'locale' => $locale
                            ]);
                        } else {
                            $columns = [];
                            $meta = $page->metas()->where('locale', $locale)->where('key', $key)->first();
                            if($meta) {
                                $value = json_decode($meta->value, true);
                                $columns = array_merge_recursive_distinct($cols, $value);
                                $meta->delete();
                                
                            } 
                            $page->meta()->create([
                                'key' => $key,
                                'value' => json_encode($columns),
                                'sort' => $meta ? $meta->sort : $sort,
                                'locale' => $locale
                            ]);
                        }
                    }
                    // foreach($cols as $col) {
                    //     foreach($col as $post) {
                           
                    //     }
                    // }
                   
                   
                    $sort = $meta ? $meta->sort + 1 : $sort+1;
                }
            }
            event(new PageUpdated($page, $attributes));
            return true;
        }

        throw new GeneralException(trans('exceptions.backend.page.update_error'));
    }

    /**
     * @return bool
     *
     * @throws GeneralException
     */
    public function delete(Model $page)
    {
        if (parent::delete($page)) {
            event(new PageDeleted($page));
            return true;
        }

        throw new GeneralException(trans('exceptions.backend.page.delete_error'));
    }

    /**
     * @throws GeneralException
     */
    public function forceDelete(Model $page)
    {
        DB::transaction(function () use ($page) {
            if (parent::forceDelete($page)) {
                $page->deleteTranslations();
                $page->metas()->delete();
                event(new PageDeleted($page));
                return true;
            }

            throw new GeneralException(trans('exceptions.backend.page.delete_error'));
        });
    }

    /**
     * @return bool
     *
     * @throws GeneralException
     */
    public function restore(Model $page)
    {
        if (parent::restore(($page))) {
            return true;
        }

        throw new GeneralException(trans('exceptions.backend.page.restore_error'));
    }

    public function getTemplates()
    {
        // $templates = Theme::getFunctionProperty(setting('theme-active', config('themes.active')), 'templates');
        $templates = Eventy::filter('regsiter_page_templates', []);
        return $templates;
    }

    /**
     * Filter the string.
     *
     * @param string $string
     *
     * @return string
     */
    public function filterString($string, $length = 63)
    {
        $string = str_replace(PHP_EOL, '', $string);

        return mb_substr(strip_tags($string), 0, $length);
    }

    public function getPagesDataForMenu($page_ids)
    {
        return $this->query()->whereIn('id', $page_ids)->get();
    }
}
