<?php

namespace App\Repositories\Backend;

use App\Models\Menu\Menu;
use App\Models\Menu\MenuItem;
use App\Repositories\Repository;
use App\Exceptions\GeneralException;
use Illuminate\Database\Eloquent\Model;

/**
 * Class MenuRepository
 * @package App\Repositories\Backend
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class MenuRepository extends Repository
{
    /**
     * Associated Repository Model
     */
    const MODEL = MenuItem::class;
    
    /**
     * @param array $data
     */
    public function  create(array $data)
    {
        $selectedMenu = Menu::where('selected', "1")->first();
        $menu = self::MODEL;
        $menu = new $menu;
        foreach (locales() as $locale) {
            foreach ($menu->translatedAttributes as $attribute) {
                if (isset($data[$attribute])) {
                    $data[$locale][$attribute] = $data[$attribute];
                }
            }
        }
        $data['menu_id'] = $selectedMenu ? $selectedMenu->id : "";
        $data['sort'] = $this->orderNumber($data['parent_id']);
        $menu->fill($data);
        if(parent::save($menu))
            return $menu;

        throw new GeneralException(trans('exceptions.backend.menu.create_error'));
    }

    public function createGroup(array $data)
    {
        $group = new Menu;
        $group->title = $data['title'];
        //$group->slug = $group->createSlug($data['title']);
        $group->selected = '0';
        $group->mega_menu = isset($data['mega-menu']) ? 1 : 0;

        if(parent::save($group))
            return $group;

        throw new GeneralException(trans('exceptions.backend.menu.update_error'));
    }


    /**
     * It save the category data in parent, child relationship.
     * @param array $input
     */
    public function updateHierarchy(array $input)
    {
        $dataCount = 1;
        $childCount = 1;
        $subchildCount = 1;
        $subsubchildCount = 1;
        foreach ($input as $data) {
            if(isset($data['children'])){
                foreach($data['children'] as $child){
                    if(isset($child['children'])){
                        foreach ($child['children'] as $subchild) {
                            if(isset($subchild['children'])){
                                foreach ($subchild['children'] as $subsubchild) {
                                    $this->query()->where('id',$subsubchild['id'])->update(['parent_id' => $subchild['id'], 'sort' => $subsubchildCount]);   
                                    $subsubchildCount++;
                                }
                            }
                            $this->query()->where('id', $subchild['id'])->update(['parent_id' => $child['id'], 'sort' => $subchildCount]);
                            $subchildCount++;
                        }
                    }
                    $this->query()->where('id', $child['id'])->update(['parent_id' => $data['id'],'sort' => $childCount]);
                    $childCount++;
                }
            }
            $this->query()->where('id',$data['id'])->update(['parent_id' => 0, 'sort' => $dataCount]);
            $dataCount++;
            
        }
        return true;

    }


    /**
     * @param Model $menu
     * @return bool
     * @throws GeneralException
     */
    public function delete(Model $menu)
    {   
        if($this->hasChild($menu->id))
            return trans('exceptions.backend.menu.parent_error');

        if (parent::delete($menu)) {
            return true;
        }

        return trans('exceptions.backend.menu.delete_error');
    }


    /**
     * Returns top level menu list
     * @param null 
     * @return collection
     */
    public function getTopLevelMenu()
    {
        $menu = Menu::where('selected', '1')->first();
        return $this->query()->select('id','title')->where('parent_id','0')->where('menu_id', $menu->id)->orderBy('sort','asc')->get();
    }

    /**
     * Returns menu collection
     * @param null
     * @return collection
     */
    public function getMenuGroup()
    {
        $group = array();
        $group['selected'] = null;
        $group['menu'] = Menu::pluck('title','id');
        $menu = Menu::select('id')->where('selected','1')->first();
        if($menu)
            $group['selected'] = $menu->id;
        
        return $group;
    }

    /**
     * @param Int $id
     */
    public function selectMenu($id)
    {
        $menu = Menu::where('selected','1')->update(['selected' => '0']);
        $update = Menu::where('id',$id)->update(['selected' => '1']);
        if($update)
            return true;

        return false;
    }

    /**
     * @param null
     */
    public function allMenu()
    {
        $menu = Menu::where('selected','1')->first();
        if($menu)
            return $this->getMenuItems($menu->id);

        return false;
    }

    /**
     * @param int $id
     */
    public function getMenuItems($id)
    {
        return $this->query()
                ->with('parent', 'children')
                ->where('parent_id', '0')
                ->where('menu_id', $id)
                ->orderBy('sort','asc')
                ->get();
    }

    /**
     * Return bool true if children present in node.
     * @param int $id
     */
    public function hasChild($id)
    {
        $menu = $this->query()->with('children')->where('id',$id)->first();
        if($menu->children->count()>0) {
            return true;
        }

        return false;
    }

    public function orderNumber($parent_id)
    {
        if($parent_id == '0') {
            $menu_items = $this->query()->where('parent_id', 0)->orderBy('id','DESC')->first();
            if($menu_items != null) {
                return (int)$menu_items->sort + 1;
            }
        }
        else {
            $menu_items = $this->query()->where('parent_id', $parent_id)->orderBy('id', 'DESC')->first();
            if ($menu_items != null) {
                return (int) $menu_items->sort + 1;
            }
        }
        return 0;
    }

}
