<?php

namespace App\Repositories\Backend;

use App\Exceptions\GeneralException;
use App\Models\Category\Category;
use App\Repositories\Repository;
use Illuminate\Database\Eloquent\Model;

/**
 * Class CategoryRepository.
 *
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class CategoryRepository extends Repository
{
    /**
     * Associated Repository Model.
     */
    const MODEL = Category::class;

    /**
     * @return bool
     *
     * @throws GeneralException
     */
    public function create(array $attributes)
    {
        try{
            $data = [
                'slug' => ($attributes['slug'] != '') ? $this->slug($this->make_slug($attributes['slug'], '-')) : $this->slug($this->make_slug($attributes['title'], '-')),
                'media_id' => ($attributes['media_id'] != '') ? $attributes['media_id'] : null,
                'status' => 1,
                'display' => (isset($attributes['display']) && $attributes['display'] != '') ? $attributes['display'] : 'default'
            ];
            if ($attributes['parent_id'] != '0') {
                $parent = $this->query()->where('id', $attributes['parent_id'])->first();
                $data['parent_id'] = $parent->id;
            }
            $model = self::MODEL;
            $model = new $model();
            foreach (locales() as $locale) {
                foreach ($model->translatedAttributes as $attribute) {
                    if (isset($attributes[$attribute])) {
                        $data[$locale][$attribute] = $attributes[$attribute];
                    }
                }
            }
            $model->fill($data);
            parent::save($model);
            return $model;
        } catch(\Exception $e) {
            throw new GeneralException(trans('exceptions.backend.category.update_error'));
        } 

    }

    public function update(Model $category, array $attributes)
    {
        $data = [
            'media_id' =>  ( isset($attributes['media_id']) && ($attributes['media_id'] != '') ) ? $attributes['media_id'] : null
        ];
        $data[$attributes['lang']] = $attributes[$attributes['lang']];
        if ($attributes['parent_id'] != '0' && $attributes['parent_id'] != null) {
            $parent = $this->query()->where('id', $attributes['parent_id'])->first();
            $data['parent_id'] = $parent->id;
        }
        if ($attributes['slug'] != $category->slug) {
            $data['slug'] = ($attributes['slug'] != '') ? $attributes['slug'] : $attributes['title']; //($input['slug'] != '') ? $this->slug($this->make_slug($input['slug'], '-')) : $this->slug($this->make_slug($input['title'], '-'));
        }
        if (parent::update($category, $data)) {
            return true;
        }

        throw new GeneralException(trans('exceptions.backend.category.update_error'));
    }

    /**
     * Return the parent menu with paginate of 10 by default.
     *
     * @param int $paginate
     */
    public function getParent($paginate = 10)
    {
        return $this->query()->select('id', 'title', 'status', 'slug', 'parent_id')->whereNull('parent_id')->orderBy('created_at', 'asc')->paginate($paginate);
    }

    public function getCategoriesWithChild($display = 'default', $paginate = 10)
    {
        return $this->query()->withDepth()->where('status', 1)->where('display', $display)->orderBy('_lft', 'ASC')->paginate($paginate);
    }

    /**
     * Return the tree of approved menu.
     *
     * @param null
     *
     * @return
     */
    public function getTree($display = 'default')
    {
        return $this->query()->whereStatus('1')->where('display', $display)->get()->toTree();
    }

    /**
     * Return the collection of parent with depth.
     *
     * @param int $id
     */
    public function getChild($id)
    {
        return $this->query()->where('parent_id', $id)->withDepth()->get();
    }

    /**
     * @return bool
     *
     * @throws GeneralException
     */
    public function delete(Model $category)
    {
        if($category->id == 1) {
            throw new GeneralException('You cannot delete this category.');
        } elseif(!$category->posts->isEmpty()) {
            throw new GeneralException('Please delete all post related to this category and try again.');
        } elseif(!$category->children->isEmpty()) {
            throw new GeneralException('Please delete all child categories of this category first and try again');
        }

        if (parent::delete($category)) {
            return true;
        }

        throw new GeneralException(trans('exceptions.backend.category.delete_error'));
    }

    /**
     * Delete menu if there is no child data.
     *
     * @return bool
     */
    public function deleteSelected($ids)
    {
        if (is_array($ids)) {
            foreach ($ids as $id) {
                $category = $this->query()->find($id);
                $posts = $category->posts()->pluck('posts.id')->toArray();
                $default_category = $this->query()->find(1);
                $default_category->posts()->attach($posts);
                $category->deleteTranslations();
                $category->delete();
            }
        }
    }

    public function slug($slug, $category_id = null)
    {
        if ($category_id != null) {
            $count = $this->query()->where('id', '<>', $category_id)->where('slug', $slug)->count();
            if ($count > 0) {
                return $this->slug($slug.'-'.rand(0, 20), $category_id);
            }
        } else {
            $count = $this->query()->where('slug', $slug)->count();
            if ($count > 0) {
                return $this->slug($slug.'-'.rand(0, 20), $category_id);
            }
        }

        return $slug;
    }

    public function make_slug($string, $seperator = '-')
    {
        return preg_replace('/\s+/u', $seperator, trim($string));
    }
}
