<?php

namespace App\Models\Category;

use Illuminate\Database\Eloquent\Model;
use Kalnoy\Nestedset\NodeTrait;
use Spatie\Feed\Feedable;
use Spatie\Feed\FeedItem;
use App\Services\Sluggable;
use App\Services\SluggableScopeHelpers;
use Illuminate\Support\Facades\DB;
use Astrotomic\Translatable\Contracts\Translatable as TranslatableContract;
use Astrotomic\Translatable\Translatable;

/**
 * Class Category.
 *
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class Category extends Model implements Feedable, TranslatableContract
{
    use NodeTrait;
    use Sluggable;
    use SluggableScopeHelpers;
    use Translatable;

    public $translatedAttributes = ['title', 'description'];

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $guarded = ['id'];

    public function __construct(array $attributes = [])
    {
        parent::__construct($attributes);
        $this->table = 'categories';
    }

    public function sluggable() : array
    {
        return [
            'source' => 'title',
        ];
    }

    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    public function scopeDisplay($query, $display = 'default')
    {
        return $query->where('display', $display);
    }

    /**
     * Return the posts.
     */
    public function posts()
    {
        return $this->belongsToMany('App\Models\Post\Post')->where('posts.active', '1');
    }

    public function toFeedItem() : FeedItem
    {
        return FeedItem::create([
            'id' => $this->id,
            'title' => $this->title,
            'summary' => $this->content(30),
            'updated' => $this->created_at,
            'link' => route('post', $this->slug),
            'author' => $this->user->name,
        ]);
    }

    public static function getFeedItems($category)
    {
        return Category::where('type', 'post')->where('status', '1')->get();
    }

    public function media()
    {
        return $this->belongsTo('App\Models\Media\Media');
    }

    public function post()
    {
        return $this->belongsToMany('App\Models\Post\Post');
    }

    public function hasFeaturedImage()
    {
        if ($this->media_id) {
            return true;
        }

        return false;
    }

    public function featuredImage($thumbnail = false)
    {
        $image = DB::table(config('media.table'))->where('id', $this->media_id)->first();
        if ($image) {
            return public_asset('storage/'.str_replace('\\', '/', $image->path));
        }

        return '';
    }

    /**
     * @return string
     * Return permalink of post
     */
    public function permalink()
    {
        return route('post', $this->slug);
    }

    /**
     * @return string Return title of post
     */
    public function title()
    {
        return $this->title;
    }
}
