<?php

namespace App\Http\Controllers\Frontend;

use App\Models\Category\Category;
use App\Models\Post\Post;
use App\Repositories\Frontend\BlogRepository;
use App\Repositories\Frontend\PageRepository;
use Artesaos\SEOTools\Facades\SEOTools;
use Illuminate\Http\Request;
use Theme;

/**
 * Class PostController.
 *
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class PostController extends FrontendController
{
    /**
     * @var BlogRepository
     */
    protected $blog;

    /**
     * @var PageRepository
     */
    protected $page;

    /**
     * PostController constructor.
     */
    public function __construct(BlogRepository $blog, PageRepository $page)
    {
        $this->blog = $blog;
        $this->page = $page;
        parent::__construct();
    }

    /**
     * @param string $slug
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function index($slug, Request $request)
    {
        // check if slug is category
        $category = Category::where('slug', $slug)->where('status', '1')->first();
        if ($category) {
            $meta = [
                'title' => ($category->meta_title != null) ? $category->meta_title : setting('site-title'),
                'description' => ($category->meta_description != null) ? preg_replace("/\[[^)]+\]/", '', $category->meta_description) : setting('meta-description'),
                'image' => url('images/logo.jpg'),
            ];
            SEOTools::setTitle($meta['title'] .' - '.setting('site-title', config('app.name')));
            SEOTools::setDescription($meta['description']);
            SEOTools::opengraph()->setUrl(url()->current());
            SEOTools::setCanonical(url()->current());
            SEOTools::opengraph()->addProperty('type', 'articles');
            SEOTools::jsonLd()->addImage(setting('site-logo'));

            return Theme::view('category', compact('meta'))->withCategory($category);
        }

        if($request->has('show') && $request->show == 'draft') {
            $post = Post::with('metas')->where('slug', $slug)->where('status', 0)->where('user_id', auth()->id())->firstOrFail();
        }else {
            $post = Post::with('metas')->where('slug', $slug)->active()->firstOrFail();
        }

        if ($post == null) {
            abort(404);
        }

        // Meta details
        $meta = [
            'title' => ($post->meta_title != null) ? $post->meta_title : setting('site-title'),
            'description' => ($post->meta_description != null) ? preg_replace("/\[[^)]+\]/", '', $post->meta_description) : setting('meta-description'),
            'image' => $post->getFeaturedImage(),
        ];

        SEOTools::setTitle($meta['title'] .' - '.setting('site-title', config('app.name')));
        SEOTools::setDescription($meta['description']);
        SEOTools::opengraph()->setUrl(url()->current());
        SEOTools::setCanonical(url()->current());
        SEOTools::opengraph()->addProperty('type', 'articles');
        SEOTools::jsonLd()->addImage($meta['image']);

        //don't remove this session
        session(['post_id' => $post->id]);

        if ($post->type == 'page') {

            // For contact form submission
            if ($request->has('larapressContactToken') && $request->has('formName')) {
                if($request->larapressContactToken != csrf_token()) {
                    return redirect()->route('post', $post->slug)->withFlashDanger('Invalid token');
                }
                $this->page->sendMail($request->except(['larapressContactToken']));
                return redirect()->back();
            }
            // End contact form submission

            if ($post->template != 'default') {
                return Theme::view($post->template)->withPage($post)->withShortcodes();
            }
            return Theme::view('page')->withPage($post)->withShortcodes();
        } elseif ($post->type == 'post') {
            $comments = [];
            $comments = $post->comments()->where('parent', null)->paginate(5);

            return Theme::view('post')->withPage($post)->withComments($comments)->withShortcodes();
        } else {

            if(Theme::viewExists($post->type)){
                return Theme::view($post->type)->withPage($post)->withShortcodes();
            }
            return Theme::view('page')->withPage($post)->withShortcodes();
        }
        abort(404);
    }

    public function getBlogs()
    {
        $meta = [
            'title' => setting('site-title'),
            'description' => setting('meta-description'),
            'image' => setting('site-logo'),
        ];

        SEOTools::setTitle($meta['title'] .' - '.setting('site-title', config('app.name')));
        SEOTools::setDescription($meta['description']);
        SEOTools::opengraph()->setUrl(url()->current());
        SEOTools::setCanonical(url()->current());
        SEOTools::opengraph()->addProperty('type', 'articles');
        SEOTools::jsonLd()->addImage($meta['image']);

        return Theme::view('posts', compact('meta'));
    }
}
