<?php
namespace App\Http\Controllers\Backend;

use Setting;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Repositories\Backend\ThemeRepository;
use App\Repositories\Backend\MenuRepository;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Rawbinn\Themes\Exceptions\FileMissingException;
use Rawbinn\Themes\Facades\Theme;

/**
 * Class ThemeController
 * @package App\Http\Controllers\Backend
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class ThemeController extends BaseController
{

    /**
     * @var ThemeRepository
     */
    protected $themes;

    /**
     * @var MenuRepository
     */
    protected $menu;

    /**
     * @param ThemeRepository $themes
     * @param MenuRepository $menu
     */
    public function __construct(ThemeRepository $themes, MenuRepository $menu)
    {
        $this->themes = $themes;
        $this->menu = $menu;
    }

    /**
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $group = $this->menu->getMenuGroup();
        $menus = $group['menu'];
        $allThemes = $this->themes->getThemes();
        $theme = $this->themes->getThemeDetails();
        return view($this->theme('themes.index'), compact('allThemes','menus','theme'));
    }

    /**
     * @param Request $request
     * @return mixed
     */
    public function update(Request $request)
    {
        $this->themes->validateTheme($request->active_theme);
        // $this->themes->disablePackages();
        Setting::set('theme-active', $request->active_theme);   
        // $this->themes->enablePackages();
        return redirect()->route('admin.themes.index')->withFlashSuccess(trans('alerts.backend.general.updated',['attribute'=>'theme settings']));
    }

    /**
     * @param object $request
     */
    public function upload(Request $request)
    {
        $this->themes->deleteAllFiles(storage_path('app/themes'));
        $file = $request->file('theme');;
        $path = $file->storeAs('themes', $file->getClientOriginalName());
       
        if($this->themes->extract(storage_path('app'.DIRECTORY_SEPARATOR.'public'.DIRECTORY_SEPARATOR.$path))) {
            return redirect()->back()->withFlashSuccess('Theme has been uploaded successfully');
        }
        return redirect()->back()->withFlashDanger('Invalid theme');
    }

    public function additionalCss(Request $request)
    {
        
    }

    public function editor(Request $request)
    {
        $contents = '';
        $theme = settings('theme-active', Theme::getActive());
        $path = Theme::getThemePath($theme);

        if($request->has('file')) {
            $request_file = $request->file;
            $file_path = $path.DIRECTORY_SEPARATOR.$request_file;
            if(File::exists($file_path)) {
                $contents = file_get_contents($file_path);
            }
        } else {
            return redirect()->route('admin.theme.editor', ['file' => 'theme.json']);
        }
        
        if(File::exists($path)) {
            $theme_files[] = [
                'name' => 'Theme JSON',
                'filename' => 'theme.json',
                'path' => $path.DIRECTORY_SEPARATOR.'theme.json'
            ];
            $theme_files[] = [
                'name' => 'Theme Functions',
                'filename' => 'functions.php',
                'path' => $path.DIRECTORY_SEPARATOR.'functions.php'
            ];
            $theme_files[] = [
                'name' => 'Theme Options',
                'filename' => 'options.php',
                'path' => $path.DIRECTORY_SEPARATOR.'options.php'
            ];

            $theme_css_path = $path.DIRECTORY_SEPARATOR.'assets'.DIRECTORY_SEPARATOR.'css';
            $view_files = File::files($theme_css_path);
            $theme_file_info['directory'] = 'Stylesheet';
            foreach($view_files as $view_file) {
                $theme_file_info['children'][] = [
                    'name' => 'Stylesheet',
                    'filename' => 'assets/css/'.$view_file->getRelativePathname(),
                    'path' => $view_file->getRealPath()
                ];
            }
            $theme_files[] = $theme_file_info;
           
            $theme_view_path = $path.DIRECTORY_SEPARATOR.'views';
            $view_files = File::files($theme_view_path);
            foreach($view_files as $view_file) {
                $theme_file_info = [];
                if($view_file->getFilename() == 'index.blade.php') {
                    $theme_file_info = [
                        'name' => 'Theme Index',
                        'filename' => 'views/index.blade.php',
                        'path' => $view_file->getRealPath()
                    ];
                } elseif($view_file->getFilename() == 'header.blade.php') {
                    $theme_file_info = [
                        'name' => 'Theme Header',
                        'filename' => 'views/header.blade.php',
                        'path' => $view_file->getRealPath()
                    ];
                } elseif($view_file->getFilename() == 'footer.blade.php') {
                    $theme_file_info = [
                        'name' => 'Theme Footer',
                        'filename' => 'views/footer.blade.php',
                        'path' => $view_file->getRealPath()
                    ];
                } elseif($view_file->getFilename() == 'page.blade.php') {
                    $theme_file_info = [
                        'name' => 'Single Page',
                        'filename' => 'views/page.blade.php',
                        'path' => $view_file->getRealPath()
                    ];
                } elseif($view_file->getFilename() == 'post.blade.php') {
                    $theme_file_info = [
                        'name' => 'Single Post',
                        'filename' => 'views/post.blade.php',
                        'path' => $view_file->getRealPath()
                    ];
                } elseif($view_file->getFilename() == 'posts.blade.php') {
                    $theme_file_info = [
                        'name' => 'Blog Post',
                        'filename' => 'views/posts.blade.php',
                        'path' => $view_file->getRealPath()
                    ];
                } elseif($view_file->getFilename() == 'search.blade.php') {
                    $theme_file_info = [
                        'name' => 'Search Results',
                        'filename' => 'views/search.blade.php',
                        'path' => $view_file->getRealPath()
                    ];
                } elseif($view_file->getFilename() == 'categories.blade.php') {
                    $theme_file_info = [
                        'name' => 'Categories Page',
                        'filename' => 'views/categories.blade.php',
                        'path' => $view_file->getRealPath()
                    ];
                } else {
                    $theme_file_info = [
                        'name' => $view_file->getFilename(),
                        'filename' => $view_file->getRelativePathname(),
                        'path' => $view_file->getRealPath()
                    ];
                }
                if(count($theme_file_info)){
                    array_push($theme_files, $theme_file_info);
                }
            }
            $directories = File::directories($theme_view_path);
            foreach($directories as $directory_path) {
                $theme_file_info = [];
                $view_files = File::files($directory_path);
                $directory_path_array = explode(DIRECTORY_SEPARATOR,$directory_path);
                $directory = end($directory_path_array);
                $theme_file_info['directory'] = $directory;
                foreach($view_files as $view_file) {
                   
                    $theme_file_info['children'][] = [
                        'name' => $view_file->getFilename(),
                        'filename' => 'views/'.$directory.'/'.$view_file->getFilename(),
                        'path' => $view_file->getRealPath()
                    ];
                    
                }
                $theme_files[] = $theme_file_info;
            }

        } else {
            $message = "Theme [{$theme}] does not exist.";
			throw new FileMissingException($message);
        }
        return view('admin.themes.editor', compact('theme_files', 'contents'));
    }

    public function updateEditor(Request $request)
    {
        if($request->has('code') && $request->has('file')) {
            try {
                $theme = settings('theme-active', Theme::getActive());
                $path = Theme::getThemePath($theme);
                $file = $path.DIRECTORY_SEPARATOR.$request->file;
                if(File::exists($file)){
                    file_put_contents($file, $request->code);
                }
            }catch(\Exception $e) {
                if($request->ajax()) {
                    return response()->json(['status' => false, 'message' => $e->getMessage()]);
                }
                return redirect()->back()->withFlashDanger($e->getMessage());
            }
            if($request->ajax()) {
                return response()->json(['status' => true, 'message' => 'Theme file has been updated successfully.']);
            }
            return redirect()->back()->withFlashSuccess('Theme file has been updated successfully.');

        }
    }
}