<?php

namespace App\Http\Controllers\Backend;

use App\Models\Slider\Slider;
use App\Models\Slider\SliderItem;
use App\Repositories\Backend\SliderRepository;
use DataTables;
use Illuminate\Http\Request;

/**
 * Class SliderController.
 *
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class SliderController extends BaseController
{
    /**
     * @var SliderRepository
     */
    protected $sliders;

    public function __construct(SliderRepository $sliders)
    {
        $this->sliders = $sliders;
    }

    /**
     * @return \Illuminate\View\View
     */
    public function index()
    {
        return view($this->theme('sliders.index'));
    }

    /**
     * @return \Illuminate\View\View
     */
    public function create()
    {
        return view($this->theme('sliders.create'));
    }

    /**
     * @return mixed
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'name' => 'required',
        ]);
        $this->sliders->create($request->all());

        return redirect()->route('admin.slider.index')->withFlashSuccess(trans('alerts.backend.general.created', ['attribute' => 'slider']));
    }

    /**
     * @return mixed
     */
    public function edit(Slider $slider, Request $request)
    {
        return view($this->theme('sliders.edit'))
            ->withSlider($slider);
    }

    /**
     * @return mixed
     */
    public function update(Slider $slider, Request $request)
    {
        $this->sliders->update($slider, $request->all());

        return redirect()->route('admin.slider.index')->withFlashSuccess(trans('alerts.backend.general.updated', ['attribute' => 'slider']));
    }

    /**
     * @param Slider slider
     *
     * @return mixed
     */
    public function destroy(Slider $slider, Request $request)
    {
        if ($this->sliders->delete($slider)) {
            return redirect()->route('admin.slider.index')->withFlashSuccess(trans('alerts.backend.general.deleted', ['attribute' => 'slider']));
        }
    }

    /**
     * @return Datatables
     */
    public function getForDatatable(Request $request)
    {
        return Datatables::of($this->sliders->getForDataTable())
            ->editColumn('created_at', function ($slider) {
                return $slider->created_at->diffForHumans();
            })
            ->editColumn('updated_at', function ($slider) {
                return $slider->updated_at->diffForHumans();
            })
            ->addColumn('actions', function ($slider) {
                return $slider->action_buttons;
            })
            ->escapeColumns([])
            ->make(true);
    }

    /**
     * @param int $id
     *
     * @return View
     */
    public function allSliderItems($id)
    {
        $slider = Slider::find($id);

        return view($this->theme('sliders.allItems'))->withSlider($slider);
    }

    /**
     * @param int $id
     *
     * @return View
     */
    public function createSliderItems($id)
    {
        $slider = Slider::find($id);

        return view($this->theme('sliders.createItems'))->withSlider($slider);
    }

    /**
     * @param int $id
     *
     * @return mixed
     */
    public function storeSliderItems($id, Request $request)
    {
        $this->validate($request, [
            'media_id' => 'required',
        ]);
        $slider = Slider::find($id);
        $sliderItem = new SliderItem();
        $sliderItem->title = $request->title;
        $sliderItem->description = $request->description;
        $sliderItem->media_id = $request->media_id;
        $sliderItem->slider_id = $slider->id;
        if ($sliderItem->save()) {
            return redirect()->route('admin.slider.item.index', $slider->id)
            ->withFlashSuccess('New slide item added successfully.');
        }

        return redirect()->route('admin.slider.item.index', $slider->id)
        ->withFlashDanger('There is some problem during inserting new slide.');
    }

    /**
     * @param int $id
     *
     * @return View
     */
    public function editSliderItems($id)
    {
        $sliderItem = SliderItem::find($id);

        return view($this->theme('sliders.editItems'))->withSliderItem($sliderItem);
    }

    /**
     * @param int $id
     *
     * @return mixed
     */
    public function updateSliderItems($id, Request $request)
    {
        $this->validate($request, [
            'media_id' => 'required',
        ]);

        $slider = Slider::find($id);
        $sliderItem = SliderItem::find($id);
        $sliderItem->title = $request->title;
        $sliderItem->description = $request->description;
        if ($request->media_id != '0') {
            $sliderItem->media_id = $request->media_id;
        }
        if ($sliderItem->save()) {
            return redirect()->back()
            ->withFlashSuccess('Slide item updated successfully.');
        }

        return redirect()->back()
        ->withFlashDanger('There is some problem during updating slide.');
    }

    /**
     * @param SliderItem sliderItem
     *
     * @return mixed
     *
     * @throws \Exception
     */
    public function destroySliderItems(SliderItem $sliderItem, Request $request)
    {
        if ($sliderItem->delete()) {
            return redirect()->back()->withFlashSuccess(trans('alerts.backend.general.deleted', ['attribute' => 'slider item']));
        }
    }

    /**
     * @return DataTables
     *
     * @throws \Exception
     */
    public function getItemsForDatatable(Request $request)
    {

        $sliderItems = SliderItem::where('slider_id', $request->id)->get();

        return DataTables::of($sliderItems)
        ->editColumn('created_at', function ($sliderItem) {
            return $sliderItem->created_at->diffForHumans();
        })
        ->addColumn('image', function ($sliderItem) {
            return '<img src="'.image($sliderItem->media_id).'" width="150px">';
        })
        ->addColumn('actions', function ($sliderItem) {
            return $sliderItem->action_buttons;
        })
        ->escapeColumns([])
        ->make(true);
    }
}
