<?php

namespace App\Http\Controllers\Backend;

use App\Http\Requests\Backend\Page\ManagePageRequest;
use App\Http\Requests\Backend\Page\StorePageRequest;
use App\Http\Requests\Backend\Page\UpdatePageRequest;
use App\Models\Post\Post;
use App\Repositories\Backend\PageRepository;
use DataTables;
use Illuminate\Http\Request;

/**
 * Class PageController.
 *
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class PageController extends BaseController
{
    /**
     * @var PageRepository
     */
    protected $pages;

    public function __construct(PageRepository $pages)
    {
        $this->pages = $pages;
    }

    /**
     * @return \Illuminate\View\View
     */
    public function index()
    {
        return view($this->theme('pages.index'));
    }

    /**
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $templates = $this->pages->getTemplates();

        return view($this->theme('pages.create'))->withTemplates($templates);
    }

    /**
     * @return mixed
     */
    public function store(StorePageRequest $request)
    {
        $page = $this->pages->create($request->all());
        if ($request->has('saveEdit')) {
            return redirect()->route('admin.page.edit', $page->id)->withFlashSuccess(trans('alerts.backend.general.created', ['attribute' => 'page']).' <a href="'.$page->permalink().'" style="color: #fff81a;" target="_blank">View Page</a>');
        }

        return redirect()->route('admin.page.index')->withFlashSuccess(trans('alerts.backend.general.created', ['attribute' => 'page']).' <a href="'.$page->permalink().'" style="color: #fff81a;" target="_blank">View Page</a>');
    }

    /**
     * @return View
     */
    public function edit(Post $page, ManagePageRequest $request)
    {
        $lang = lang($request->lang);
        $templates = $this->pages->getTemplates();

        return view($this->theme('pages.edit'))
            ->withPage($page)
            ->withLang($lang)
            ->withTemplates($templates);
    }

    /**
     * @return mixed
     */
    public function update(Post $page, UpdatePageRequest $request)
    {
        $this->pages->update($page, $request->all());

        $locale = $this->lang ?? app()->getLocale();
        return redirect()->back()->withFlashSuccess(trans('alerts.backend.general.updated', ['attribute' => 'page']).' <a href="'.$page->permalink().'" style="color: #fff81a;" target="_blank">View Page</a>');
    }

    /**
     * @param Post page
     *
     * @return mixed
     */
    public function destroy(Post $page, ManagePageRequest $request)
    {
        if ($this->pages->delete($page)) {
            return redirect()->route('admin.page.index')->withFlashSuccess(trans('alerts.backend.general.deleted', ['attribute' => 'page']));
        }
    }

    public function destroySelected(Request $request)
    {
        if (!$request->ajax()) {
            exit;
        }
        foreach ($request->id as $id) {
            $page = Post::find($id);
            if ($page != null) {
                $page->delete();
            }
        }

        return response()->json(['status' => true]);
    }

    /**
     * @param null
     *
     * @return view
     */
    public function deleted()
    {
        return view($this->theme('pages.deleted'));
    }

    /**
     * @return mixed
     */
    public function delete(Post $deletedPage, ManagePageRequest $request)
    {
        $this->pages->forceDelete($deletedPage);

        return redirect()->route('admin.page.deleted')->withFlashSuccess(trans('alerts.backend.general.deleted_permanently', ['attribute' => 'page']));
    }

    public function deleteSelected(Request $request)
    {
        if (!$request->ajax()) {
            exit;
        }
        if (!$request->has('id')) {
            return response()->json(['status' => false]);
        }
        foreach ($request->id as $id) {
            $page = Post::withTrashed()->where('id', $id)->first();
            if ($page != null) {
                $this->pages->forceDelete($page);
            }
        }

        return response()->json(['status' => true]);
    }

    /**
     * @return mixed
     */
    public function restore(Post $deletedPage, ManagePageRequest $request)
    {
        $this->pages->restore($deletedPage);

        return redirect()->route('admin.page.index')->withFlashSuccess(trans('alerts.backend.general.restored', ['attribute' => 'page']));
    }

    /**
     * @return mixed
     */
    public function getForDatatable(ManagePageRequest $request)
    {
        return DataTables::of($this->pages->getForDataTable($request->get('trashed')))
            ->editColumn('title', function ($page) {
                return '<a href="'.route('post', $page->slug).'" target="_blank">'.$page->title.'</a>';
            })
            ->addColumn('author', function ($page) {
                return $page->user->name;
            })
            ->editColumn('created_at', function ($page) {
                return $page->created_at->diffForHumans();
            })
            ->editColumn('updated_at', function ($page) {
                return $page->updated_at->diffForHumans();
            })
            ->addColumn('actions', function ($page) {
                return $page->action_buttons;
            })
            ->addColumn('select', function ($page) {
                return '<input type="checkbox" name="id[]" value="'.$page->id.'">';
            })
            ->escapeColumns([])
            ->make(true);
    }
}
