<?php

namespace App\Http\Controllers\Backend;

use App\Repositories\Backend\MediaRepository;
use DB;
use Illuminate\Http\Request;
use Intervention\Image\Facades\Image;
use Rawbinn\Media\Models\Media;

/**
 * Class MediaController.
 *
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class MediaController extends BaseController
{
    /**
     * @var MediaRepository
     */
    protected $media;

    public function __construct(MediaRepository $media)
    {
        $this->media = $media;
    }

    public function index(Request $request)
    {
        if ($request->has('type') && $request->type != '' && $request->has('date') && $request->date != '') {
            $date = explode('-', $request->date);
            $date = count($date) > 1 ? [$date[0], $date[1]] : [$date[0]];
            if ($request->type == 'image') {
                $files = $this->media->type('image')->date($date)->paginate(35);
            } elseif ($request->type == 'pdf') {
                $files = $this->media->type('file')->extension('pdf')->date($date)->paginate(35);
            } elseif ($request->type == 'video') {
                $files = $this->media->type('file')->extension('mp4')->date($date)->paginate(35);
            } elseif ($request->type == 'audio') {
                $files = $this->media->type('file')->extension('mp3')->date($date)->paginate(35);
            } elseif ($request->type == 'doc') {
                $files = $this->media->type('file')->extension('doc')->date($date)->paginate(35);
            } else {
                $files = $this->media->date($date)->paginate(35);
            }
        } elseif ($request->has('type') && $request->type != '') {
            if ($request->type == 'image') {
                $files = $this->media->type('image')->paginate(35);
            } elseif ($request->type == 'pdf') {
                $files = $this->media->type('file')->extension('pdf')->paginate(35);
            } elseif ($request->type == 'doc') {
                $files = $this->media->type('file')->extension('doc')->paginate(35);
            } elseif ($request->type == 'video') {
                $files = $this->media->type('file')->extension('mp4')->paginate(35);
            } elseif ($request->type == 'audio') {
                $files = $this->media->type('file')->extension('mp3')->paginate(35);
            } else {
                $files = $this->media->getPaginated(35);
            }
        } elseif ($request->has('date') && $request->date != '') {
            $date = explode('-', $request->date);
            $date = count($date) > 1 ? [$date[0], $date[1]] : [$date[0]];
            $files = $this->media->date($date)->paginate(35);
        } else {
            $files = $this->media->getPaginated(35);
        }
        $dates = $this->media->dateForOption();

        return view($this->theme('media.index'))->withFiles($files)->withDates($dates);
    }

    public function create()
    {
        return view($this->theme('media.create'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required',
            'file' => 'max:10000|mimes:pdf,doc,docx',
        ]);
        $uploadedFile = $request->file('file');
        $filename = date('Ymd').time().'.'.$uploadedFile->getClientOriginalExtension();
        try {
            $path = $uploadedFile->storeAs('files', $filename, 'public');

            Media::create([
                'user_id' => auth()->id(),
                'title' => $request->title,
                'image' => $filename,
                'path' => $path,
                'type' => 'file',
                'extension' => $uploadedFile->getClientOriginalExtension(),
            ]);
        } catch (\Exception $e) {
            return redirect()->back()->withFlashDanger($e->getMessage());
        }

        return redirect()->back()->withFlashSuccess(trans('alerts.backend.general.created', ['attribute' => 'media']));
    }

    public function edit($id)
    {
        $media = DB::table('media')->where('id', $id)->first();

        return view($this->theme('media.edit'))->withMedia($media);
    }

    public function update($id, Request $request)
    {
        $request->validate([
            'title' => 'required',
            'file' => 'sometimes|max:10000|mimes:pdf,doc,docx',
        ]);
    }

    public function delete($id)
    {
        $media = Media::where('id', $id)->first();
        try {
            unlink(public_path('storage/'.$media->path));
            $media->delete();
        }catch(\Exception $e) {
                return response()->json(['message' => $e->getMessage()], 500);
        }

        return response()->json(['message' => 'Image deleted successfully.']);
    }

    public function mediaDetails(Request $request)
    {
        $media = [];
        $file = $this->media->find($request->id);
        if ($file->type == 'image') {
            $image = Image::make(public_path('storage/'.$file->path));
            $size = $image->filesize();
            $dimension = $image->width().' x '.$image->height().' pixels';
            $filetype = $image->mime();
            $media = [
                'title' => $file->title,
                'caption' => $file->caption,
                'alt' => $file->file_original_name,
                'image' => $file->id,
                'path' => $file->url,
                'type' => $file->type,
                'file_size' => $size / 100 .' KB',
                'extension' => $file->extension,
                'dimension' => $dimension,
                'filetype' => $filetype,
                'uploaded_by' => $file->user->name,
                'created_at' => $file->created_at->format('M d, Y'),
            ];
        } else {
        }

        return response()->json($media);
    }

    public function updateDetails($id, Request $request)
    {
        if ($request->has('title')) {
            Media::where('id', $id)->update(['title' => $request->title]);
        }
        if ($request->has('caption')) {
            Media::where('id', $id)->update(['caption' => $request->caption]);
        }

        return 'OK';
    }
}
