<?php

namespace App\Http\Controllers\Backend;

use App\Events\Backend\Blog\BlogDeleted;
use App\Http\Requests\Backend\Blog\DeleteBlogRequest;
use App\Http\Requests\Backend\Blog\EditBlogRequest;
use App\Http\Requests\Backend\Blog\ManageBlogRequest;
use App\Http\Requests\Backend\Blog\StoreBlogRequest;
use App\Http\Requests\Backend\Blog\UpdateBlogRequest;
use App\Models\Post\Post;
use App\Repositories\Backend\BlogRepository;
use App\Repositories\Backend\CategoryRepository;
use DataTables;
use Illuminate\Http\Request;

/**
 * Class BlogController.
 *
 * @author Rawbinn Shrestha <rawbinnn@gmail.com>
 */
class BlogController extends BaseController
{
    /**
     * @var BlogRepository
     */
    protected $blogs;

    /**
     * @var CategoryRepository
     */
    protected $category;

    /**
     * BlogController constructor.
     */
    public function __construct(BlogRepository $blogs, CategoryRepository $category)
    {
        $this->blogs = $blogs;
        $this->category = $category;
    }

    /**
     * @return \Illuminate\View\View
     */
    public function index()
    {
        return view($this->theme('blogs.index'));
    }

    /**
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $categories = $this->category->getTree(); // Returns tree of approved categories

        return view($this->theme('blogs.create'), compact('categories'));
    }

    /**
     * @param Request $request
     *
     * @return mixed
     */
    public function store(StoreBlogRequest $request)
    {
        $post = $this->blogs->create($request->all());
        if ($request->has('saveEdit')) {
            return redirect()->route('admin.blog.edit', $post->id)->withFlashSuccess(trans('alerts.backend.general.created', ['attribute' => 'blog']).' <a href="'.$post->permalink().'" style="color: #fff81a;" target="_blank">View Post</a>');
        }

        return redirect()->route('admin.blog.index')->withFlashSuccess(trans('alerts.backend.general.created', ['attribute' => 'blog']).' <a href="'.$post->permalink().'" style="color: #fff81a;" target="_blank">View Post</a>');
    }

    /**
     * @param EditBlogRequest $request
     *
     * @return View
     */
    public function edit(Post $post, EditBlogRequest $request)
    {
        $lang = lang($request->lang);
        $categories = $this->category->getTree(); // Returns tree of approved categories
        return view($this->theme('blogs.edit'), compact('categories', 'lang', 'post'));
    }

    /**
     * @return mixed
     */
    public function update(Post $post, UpdateBlogRequest $request)
    {
        $this->blogs->update($post, $request->all());
        $permalink = $post->status == 0 ? $post->permalink().'?show=draft' : $post->permalink();
        return redirect()->back()->withFlashSuccess(trans('alerts.backend.general.updated', ['attribute' => 'blog']).' <a href="'.$permalink.'" style="color: #fff81a;" target="_blank">View Post</a>');
    }

    /**
     * @param Post blog
     *
     * @return mixed
     */
    public function destroy(Post $post, DeleteBlogRequest $request)
    {
        if ($this->blogs->delete($post)) {
            return redirect()->back()->withFlashSuccess(trans('alerts.backend.general.deleted', ['attribute' => 'blog']));
        }
    }

    public function destroySelected(Request $request)
    {
        if (!$request->ajax()) {
            exit;
        }
        foreach ($request->id as $id) {
            $post = Post::find($id);
            if ($post != null) {
                $post->delete();
            }
        }
        event(new BlogDeleted($post));

        return response()->json(['status' => true]);
    }

    /**
     * @param null
     *
     * @return view
     */
    public function deleted()
    {
        return view('admin.blogs.deleted');
    }

    /**
     * @return mixed
     */
    public function delete(Post $deletedBlog, ManageBlogRequest $request)
    {
        $this->blogs->forceDelete($deletedBlog);

        return redirect()->route('admin.blog.deleted')->withFlashSuccess(trans('alerts.backend.general.deleted_permanently', ['attribute' => 'blog']));
    }

    public function deleteSelected(Request $request)
    {
        if (!$request->ajax()) {
            exit;
        }
        if (!$request->has('id')) {
            return response()->json(['status' => false]);
        }
        foreach ($request->id as $id) {
            $post = Post::withTrashed()->where('id', $id)->first();
            if ($post != null) {
                $this->blogs->forceDelete($post);
            }
        }

        return response()->json(['status' => true]);
    }

    /**
     * @return mixed
     */
    public function restore(Post $deletedBlog, ManageBlogRequest $request)
    {
        $this->blogs->restore($deletedBlog);

        return redirect()->route('admin.blog.index')->withFlashSuccess(trans('alerts.backend.general.restored', ['attribute' => 'blog']));
    }

    /**
     * @return DataTables
     */
    public function getForDatatable(ManageBlogRequest $request)
    {
        return DataTables::of($this->blogs->getForDataTable($request->get('trashed')))
            ->editColumn('title', function ($blog) {
                return '<a href="'.route('post', $blog->slug).'" target="_blank">'.$blog->title.'</a>';
            })
            ->addColumn('author', function ($blog) {
                return $blog->user->name;
            })
            ->addColumn('category', function ($blog) {
                $categories = [];
                foreach ($blog->categories as $category) {
                    $categories[]= $category->title;
                }

                return implode(',', $categories);
            })
            ->editColumn('status', function ($blog) {
                return $blog->status == 1 ? 'Published' : 'Draft';

                return ($block->status == 1) ? '<span class="text-success">Published</span>' : '<span class="text-danger">Draft</span>';
            })
            ->editColumn('updated_at', function ($blog) {
                return $blog->updated_at->diffForHumans();
            })
            ->addColumn('actions', function ($blog) {
                return $blog->action_buttons;
            })
            ->addColumn('select', function ($blog) {
                return '<input type="checkbox" name="id[]" value="'.$blog->id.'">';
            })
            ->escapeColumns([])
            ->make(true);
    }
}
