<?php

namespace App\Http\Controllers\Backend\Access\User;

use App\Exceptions\GeneralException;
use App\Http\Controllers\Backend\BaseController;
use App\Models\Access\User\User;
use App\Http\Controllers\Controller;
use App\Repositories\Backend\Access\User\UserRepository;
use App\Repositories\Backend\Access\Role\RoleRepository;
use App\Http\Requests\Backend\Access\User\StoreUserRequest;
use App\Http\Requests\Backend\Access\User\ManageUserRequest;
use App\Http\Requests\Backend\Access\User\UpdateUserRequest;
use Illuminate\Http\Request;

/**
 * Class UserController
 */
class UserController extends BaseController
{
    /**
     * @var UserRepository
     */
    protected $users;

    /**
     * @var RoleRepository
     */
    protected $roles;

    /**
     * @param UserRepository $users
     * @param RoleRepository $roles
     */
    public function __construct(UserRepository $users, RoleRepository $roles)
    {
        $this->users = $users;
        $this->roles = $roles;
    }

	/**
     * @param ManageUserRequest $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(ManageUserRequest $request)
    {
        return view($this->theme('access.index'));
    }

	/**
     * @param ManageUserRequest $request
     * @return mixed
     */
    public function create(ManageUserRequest $request)
    {
        return view($this->theme('access.create'))
            ->withRoles($this->roles->getAll());
    }

	/**
     * @param StoreUserRequest $request
     * @return mixed
     */
    public function store(StoreUserRequest $request)
    {
        $this->users->create(['data' => $request->except('assignees_roles'), 'roles' => $request->only('assignees_roles')]);
        return redirect()->route('admin.access.user.index')->withFlashSuccess(trans('alerts.backend.users.created'));
    }

	/**
	 * @param User $user
	 * @param ManageUserRequest $request
	 * @return mixed
	 */
	public function show(User $user, ManageUserRequest $request) {
		return view($this->theme('access.show'))
			->withUser($user);
	}

	/**
     * @param User $user
     * @param ManageUserRequest $request
     * @return mixed
     */
    public function edit(User $user, ManageUserRequest $request)
    {
        return view($this->theme('access.edit'))
            ->withUser($user)
            ->withUserRoles($user->roles->pluck('id')->all())
            ->withRoles($this->roles->getAll());
    }

	/**
     * @param User $user
     * @param UpdateUserRequest $request
     * @return mixed
     */
    public function update(User $user, UpdateUserRequest $request)
    {
        $this->users->update($user, ['data' => $request->except('assignees_roles'), 'roles' => $request->only('assignees_roles')]);
        return redirect()->route('admin.access.user.index')->withFlashSuccess(trans('alerts.backend.users.updated'));
    }

	/**
     * @param User $user
     * @param ManageUserRequest $request
     * @return mixed
     */
    public function destroy(User $user, ManageUserRequest $request)
    {
        $this->users->delete($user);
        return redirect()->route('admin.access.user.deleted')->withFlashSuccess(trans('alerts.backend.users.deleted'));
    }

    public function profile(Request $request)
    {
        $user = auth()->user();
        if($request->isMethod('post')) {
            $request->validate([
                'name' => 'required',
                'email' => 'required|unique:users,email,'.$user->id,
                'gender' => 'required|in:male,female,other'
            ]);

            try{
                $user->email = $request->email;
                $user->save();

                $fullname = explode(' ', $request->name);
                $profile = $user->profile;
                $profile->firstname = $fullname[0];
                $profile->lastname = $fullname[1] ?? '';
                $profile->gender = strtolower($request->gender);
                $profile->mobile = $request->mobile_number;
                $profile->address = $request->address;
                $profile->about = $request->about;
                $profile->profile_picture = $request->profile_picture;
                $profile->save();
            } catch(\Exception $e) {
                throw new GeneralException($e->getMessage());
            }
            return redirect()->back()->withFlashSuccess('Profile information has been updated successfully.');
        }        

        return view('admin.access.profile', compact('user'));
    }
}